#!/usr/bin/env python2
# -*- coding: utf-8 -*-
"""
Created on Sun Mar 11 14:29:01 2018

@author: mischaknabenhans
"""

#%% Import modules
import numpy as np
import matplotlib.pyplot as plt

#%% Auxiliary equations

# Hamiltonian of harmonic oscillator
def HO_Hamiltonian(q,p):
    return 0.5*(q**2 + p**2)

# Foreward Euler
def fwEuler(q,p,h):
    return [q+h*p, p-h*q]   # Hamilton equations

# leap frog
def leapFrog(q,p,h):
    q_half = q + 0.5*h*p
    p_new  = p - h*q_half
    q_new  = q_half + 0.5*h*p_new
    
    return [q_new, p_new]

def SolveHamiltonEvolution(q,p,h,method):
    qList = [q]
    pList = [p]
    EList = [2*HO_Hamiltonian(q,p)]

    # Compute phase space evolution
    for c in range(nSteps):
        if method=='Euler':
            [q, p] = fwEuler(q,p,h)
        elif method=='LeapFrog':
            [q, p] = leapFrog(q,p,h)
        qList.append(q)
        pList.append(p)
        EList.append(2*HO_Hamiltonian(q,p))
    
    qvec = np.array(qList)
    pvec = np.array(pList)
    Evec = np.array(EList)

    return [qvec, pvec, Evec]

#%% Main
    
t_ini = 0
t_fin = 20
nSteps = int(1e3)

h = float(t_fin-t_ini)/nSteps
tvec = np.linspace(t_ini, t_fin, nSteps+1) #the boundaries are included

# Initial conditions
q_fE_Arr = []
p_fE_Arr = []
E_fE_Arr = []

q_lf_Arr = []
p_lf_Arr = []
E_lf_Arr = []

Elevels = [0.5,1,2]
for E in Elevels:
    q = 1
    p = np.sqrt(E/0.5-q**2)

    # Compute phase space evolution
    [qvec_fE, pvec_fE, Evec_fE]=SolveHamiltonEvolution(q,p,h,'Euler')
    [qvec_lf, pvec_lf, Evec_lf]=SolveHamiltonEvolution(q,p,h,'LeapFrog')

    q_fE_Arr.append(qvec_fE)
    p_fE_Arr.append(pvec_fE)
    E_fE_Arr.append(Evec_fE)

    q_lf_Arr.append(qvec_lf)
    p_lf_Arr.append(pvec_lf)
    E_lf_Arr.append(Evec_lf)

# Plot
E_label = ['0.5','1.0','2.0']    
    
Fig1, axs = plt.subplots(2,2)

# Position, momentum & energy vs time
ax = axs[0,0]
ax.plot(tvec, qvec_fE, label="q(t)")
ax.plot(tvec, pvec_fE, label="p(t)")
ax.plot(tvec, Evec_fE, label="E(t)")
ax.set_xlim([-1,31])
ax.legend(loc="center right")

# Phase space plot
ax = axs[0,1]
for i in range(len(Elevels)):
    ax.plot(q_fE_Arr[i],p_fE_Arr[i], label="E="+E_label[i])
ax.set_ylabel("p")
ax.legend(loc="best")
ax.set_aspect('equal', 'datalim')


ax = axs[1,0]
ax.plot(tvec, qvec_lf, label="q(t)")
ax.plot(tvec, pvec_lf, label="p(t)")
ax.plot(tvec, Evec_lf, label="E(t)")
ax.set_xlabel("t")
ax.set_xlim([-1,31])
ax.legend(loc="center right")

# Phase space plot
ax = axs[1,1]
for i in range(len(Elevels)):
    ax.plot(q_lf_Arr[i],p_lf_Arr[i], label="E="+E_label[i])
ax.set_xlabel("q")
ax.set_ylabel("p")
ax.legend(loc="best")
ax.set_aspect('equal', 'datalim')

#Fig1.suptitle("foreward Euler (top) vs. leap frog (bottom)", y=1.01)
Fig1.tight_layout()
Fig1.show()




    