# Mandelbrot set with Tkinter
# We use an image that is put into the canvas because drawing individual rectangles
# or points (lines of length 1) needs very much time (several minutes)!
# The image is redrawn automatically when the image data is updated.

import Tkinter as tk, threading, math, time, random, tkFont
import colormap

WINDOW_WIDTH = 400
WINDOW_HEIGHT = 400
    
def main():
    app = Application()
    app.master.title('Mandelbrot')
    task = Mandelbrot(app.canvas)
    app.task = task
    task.start()
    app.mainloop()
    task.isRunning = False
    task.join()
    print 'exit'
    
class Application(tk.Frame):
    def __init__(self, master=None):
        tk.Frame.__init__(self, master)
        self.grid()
        self.createWidgets()
    
    def handleResetButton(self):
        self.task.RE_MIN = -1.6
        self.task.RE_MAX = 0.6
        self.task.IM_MIN = -1.0
        self.task.IM_MAX = 1.0
        self.task.do_recalculate = True
        
    def createWidgets(self):
        self.canvas = tk.Canvas(self, bg="white", width=WINDOW_WIDTH, height=WINDOW_HEIGHT)
        self.canvas.grid(row=1)
        self.resetButton = tk.Button(self, text='Reset', command=self.handleResetButton)
        self.resetButton.grid(row=2, sticky=tk.SW+tk.E+tk.W)
        self.quitButton = tk.Button(self, text='Quit', command=self.quit)
        self.quitButton.grid(row=2, sticky=tk.NE)
        self.label = tk.Label(self, text='Press mouse button-1 and drag mouse to zoom in')
        self.label.grid(row=3)
        
class Mandelbrot(threading.Thread):
        
    N = WINDOW_WIDTH
    MAX_ITER = 100
    IM_MIN, IM_MAX = -1.0, 1.0
    RE_MIN, RE_MAX = -1.6, 0.6
    #RE_MIN, RE_MAX = -0.75, -0.74
    #IM_MIN, IM_MAX = 0.16, 0.17
    do_recalculate = True
    x0 = 0
    y0 = 0
    buttonDown = False
    isRunning = True
        
    def __init__(self, canvas):
        threading.Thread.__init__(self)
        self.canvas = canvas
        canvas.bind('<Button-1>', self.buttonPress)
        canvas.bind('<ButtonRelease-1>', self.buttonRelease)
        canvas.bind('<Motion>', self.mouseMove)
        self.rect = canvas.create_rectangle(0, 0, 0, 0)
        
    def buttonPress(self, event):
        self.x0, self.y0 = event.x, event.y
        self.buttonDown = True
        
    def buttonRelease(self, event):
        x1, y1 = event.x, event.y
        self.buttonDown = False
        
        x_start = min(self.x0, x1)
        x_end = max(self.x0, x1)
        y_start = min(self.y0, y1)
        y_end = max(self.y0, y1)
        im_size = self.IM_MAX - self.IM_MIN
        re_size = self.RE_MAX - self.RE_MIN
        im_min = self.IM_MIN
        re_min = self.RE_MIN
        self.RE_MIN = re_min + x_start * re_size / self.N
        self.RE_MAX = re_min + x_end * re_size / self.N
        self.IM_MIN = im_min + y_start * im_size / self.N
        self.IM_MAX = im_min + y_end * im_size / self.N
        #bounds = (self.IM_MIN, self.IM_MAX, self.RE_MIN, self.RE_MAX)
        #print bounds
        self.do_recalculate = True
        
    def mouseMove(self, event):
        if self.buttonDown:
            self.canvas.delete(self.rect)
            self.rect = self.canvas.create_rectangle(self.x0, self.y0, event.x, event.y, outline='white')
        
    def iterate(self, re, im, MAX_ITER):
        C = re + im * 1j
        z = 0
        n_iter = 0
        while n_iter < MAX_ITER:
            z = z**2 + C
            if abs(z) > max(2.0, abs(C)):
                break # escapee
            n_iter += 1
                     
        if n_iter == MAX_ITER:
            # current point is a prisoner
            rgb = [0, 0, 0]
        else:
            # current point is an escapee
            #rgb = colormap.rainbow_colormap(1.0 * n_iter / MAX_ITER)
            rgb = colormap.iron_colormap(1.0 * n_iter / MAX_ITER)

        return colormap.to_string(rgb)
        
    def calculate(self, bounds, N, MAX_ITER):
        im_min, im_max, re_min, re_max = bounds
    
        # Create an image and add it to the canvas. The image is redrawn when
        # the image data is updated later on.
        image = tk.PhotoImage(width=N, height=N)

        # NOTE: Store a reference to the image as an attribute of the widget
        # to avoid garbage collection!!!
        self.canvas.img = image
        self.canvas.create_image(N/2, N/2, image=image)   
        
        im_list = [im_min + i*(im_max - im_min) / N for i in range(N)]
        re_list = [re_min + i*(re_max - re_min) / N for i in range(N)]
       
        y = 0
        for im in im_list:
            color_list = ["{"]
            color_list.extend([self.iterate(re, im, MAX_ITER) for re in re_list])
            color_list.append("}")
            colors = " ".join(color_list)
            image.put(colors, (0,y))
            y += 1
            
    def calculate_loop(self, bounds, N, MAX_ITER):
        im_min, im_max, re_min, re_max = bounds
    
        # Create an image and add it to the canvas. The image is redrawn when
        # the image data is updated later on.
        image = tk.PhotoImage(width=N, height=N)

        # NOTE: Store a reference to the image as an attribute of the widget
        # to avoid garbage collection!!!
        self.canvas.img = image
        self.canvas.create_image(N/2, N/2, image=image)   
          
        im = im_min
        for y in range(N):
            re = re_min
            color_list = ['{']
            for x in range(N):
                C = re + im * 1j
                z = 0
                n_iter = 0
                while n_iter < MAX_ITER:
                    z = z**2 + C
                    if abs(z) > max(2.0, abs(C)):
                        break # escapee
                    n_iter += 1

                if n_iter == MAX_ITER:
                    # current point is a prisoner
                    rgb = [0, 0, 0]
                else:
                    # current point is an escapee
                    #rgb = colormap.rainbow_colormap(1.0 * n_iter / MAX_ITER)
                    rgb = colormap.iron_colormap(1.0 * n_iter / MAX_ITER)

                color = colormap.to_string(rgb)
                #image.put(color, (x,y))
                color_list.append(color)
 
                # Drawing individual rectangles or points needs very much time (several minutes)!
                #coords = x, y, x+1, y+1
                #self.canvas.create_rectangle(coords, fill=color, outline=color)
                #coords = x, y, x+1, y
                #self.canvas.create_line(coords, fill=color)

                re += (re_max - re_min) / N
            color_list.append('}')
            image.put(" ".join(color_list), (0,y))  # more efficient than putting single color values              
            im += (im_max - im_min) / N
            
    def run(self):
        while self.isRunning:
            if self.do_recalculate == True:
                bounds = (self.IM_MIN, self.IM_MAX, self.RE_MIN, self.RE_MAX)
                self.calculate(bounds, self.N, self.MAX_ITER)
                self.do_recalculate = False
            time.sleep(0.1)

if __name__ == '__main__':
    main()

